/*
SWR Meter by Glen Popiel - KW5GP

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 3 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

  Connections for SWR RF head
  Forward connected to Analog 0
  Reflected connected to Analog 1

  Nokia 5110 LCD module is connected to the following pins.
      CLK  - Pin 12
      DIN  - Pin 11
      DC   - Pin 10
      CE   - Pin 9
      RST  - Pin 8
*/

#include <LCD5110_Basic.h>  // Include Nokia 5110 LCD Library

LCD5110 glcd(12,11,10,8,9);  // Define the LCD object

extern uint8_t SmallFont[];  // Define the LCD Font

#include <SoftwareSerial.h>  // include the SoftwareSerial library so we can use it to talk to the Emic 2 module

#define rxPin 2    // Serial input (connects to Emic 2 SOUT)
#define txPin 3    // Serial output (connects to Emic 2 SIN)
#define audio_on 7 // Audio On Switch Pin

SoftwareSerial emicSerial =  SoftwareSerial(rxPin, txPin);  // set up a new serial port

#define adc_count 0.0048828125  // Volts per ADC count (0 to 5 volts, 0 to 1023 A/D counts)

float V_Fwd, V_Ref = 0, V_SWR ;  // Define the variables

void setup()
{
  // define the pin modes
  pinMode(rxPin, INPUT);
  pinMode(txPin, OUTPUT);
  pinMode(audio_on, INPUT);
  digitalWrite(audio_on, HIGH); // Enable pullup resistor
  
  emicSerial.begin(9600);  // set the data rate for the SoftwareSerial port
   
  glcd.InitLCD(60);  // Initialize the Nokia 5110 Display - set the contrast to 60
  glcd.setFont(SmallFont);

  // Display the Startup screen
  glcd.clrScr();
  glcd.print("KW5GP", CENTER,0);
  glcd.print("SWR Meter",CENTER,8);
  delay(3000);
  glcd.clrScr();
  
  emicSerial.print('\n');             // Send a CR in case the system is already up
  while (emicSerial.read() != ':');   // When the Emic 2 has initialized and is ready, it will send a single ':' character, so wait here until we receive it
  delay(10);                          // Short delay
  emicSerial.println("n1");   // Set voice to Voice 1
  delay(500);
  emicSerial.print('v');    // Set the volume to +18db
  emicSerial.println("18");
  emicSerial.flush();                 // Flush the receive buffer


}  // End Setup Loop

void loop()    // Start the Main Loop
{
  glcd.print("Fwd:",0,0);  // Display the SWR information on the LCD
  glcd.print("Ref:",0,8);
  V_Fwd = analogRead(0) * adc_count;    // Read the A/D converters and convert to voltages
  V_Ref = analogRead(1) * adc_count;
  glcd.printNumF(V_Fwd,7,30,0);  // display the Forward and Reflected voltages
  glcd.printNumF(V_Ref,7,30,8);
  
  // Calculate VSWR
  if (V_Fwd > V_Ref)
  {
    V_SWR = (V_Fwd + V_Ref) / (V_Fwd - V_Ref);
  } else {
    V_SWR = 0;
  }
  
  // Display the VSWR
  glcd.print("SWR: ",0,24);
  glcd.print("          ",30,24);
  glcd.printNumF(V_SWR,1,30,24);
  glcd.print(" : 1",56,24);
  
  if (digitalRead(audio_on) == LOW)
  {
    emicSerial.print('S');
    emicSerial.print("S W R is ");  // Send the desired string to convert to speech
    emicSerial.print(V_SWR,1);
    emicSerial.print(" to 1");
    emicSerial.print('\n');
    while (emicSerial.read() != ':');   // Wait here until the Emic 2 responds with a ":" indicating it's ready to accept the next command
  } 

  delay(1000);
 
}  // End Main Loop

